/*
 * $QNXLicenseC:
 * Copyright 2010, QNX Software Systems. All Rights Reserved.
 *
 * You must obtain a written license from and pay applicable
 * license fees to QNX Software Systems before you may reproduce,
 * modify or distribute this software, or any work that includes
 * all or part of this software.   Free development licenses are
 * available for evaluation and non-commercial purposes.  For more
 * information visit http://licensing.qnx.com or email
 * licensing@qnx.com.
 *
 * This file may contain contributions from others.  Please review
 * this entire file for other proprietary rights or license notices,
 * as well as the QNX Development Suite License Guide at
 * http://licensing.qnx.com/license-guide/ for other information.
 * $
 */

#include <string.h>
#include <errno.h>

#include "touch.h"

#ifndef UNITTEST
#include <sys/neutrino.h>
#include "input/mtouch_driver.h"
#include <input/mtouch_log.h>
#include "gpio_client.h"
#else
#include "timer_header.h"
#include "gpio_header.h"
#endif

int
tp_intr_attach(cypress_dev_t *cypress, struct sigevent *intrevent, int intr, int pulse_code)
{
    int iid = -1;

    SIGEV_PULSE_INIT(intrevent, cypress->coid, cypress->pulse_prio, pulse_code, 0);

    if (cypress->polling_rate) {
        struct itimerspec       itime;

        if (cypress->verbose)
            mtouch_info (cypress->log_name, "Using polling mode");

        timer_create (CLOCK_REALTIME, intrevent, &cypress->timer_id);

        itime.it_value.tv_sec = 0;
        itime.it_value.tv_nsec = cypress->polling_rate * MSEC;
        itime.it_interval.tv_sec = 0;
        itime.it_interval.tv_nsec = cypress->polling_rate * MSEC;
        timer_settime(cypress->timer_id, 0, &itime, NULL);
    } else {
        if (-1 == (iid = InterruptAttachEvent(intr, intrevent, _NTO_INTR_FLAGS_TRK_MSK | _NTO_INTR_FLAGS_PROCESS))){
            mtouch_error(cypress->log_name, "InterruptAttach: %s", strerror(errno));
            error_memory("Cypress_Touch: InterruptAttach: %s", strerror(errno));}
        else
            mtouch_info(cypress->log_name, "Attached to device with interrupt %d", intr);
    }

    return iid;
}

int tp_intr_gpio_to_irq(cypress_dev_t *cypress)
{
    uint32_t gpio_number = cypress->intr_gpio_pin;
    uint32_t *irq = &cypress->tp_intr;
    uint32_t gpio_number_tlmm;
    uint32_t gpio_config;
    int fd;

    if (-1 == (fd = gpio_open(NULL))) {
        mtouch_error(cypress->log_name,"gpio_open() failed");
        error_memory("Cypress_Touch: gpio_open() failed");
        goto exit_err;
    }

    if(gpio_number  <= 0 ){
        mtouch_error(cypress->log_name,"Bad GPIO input param gpio=%d \n",gpio_number);
        error_memory("Cypress_Touch: Bad GPIO input param gpio=%d \n",gpio_number);
        goto exit_err;
    }

    gpio_number_tlmm = gpio_number & 0xFFFF; // for TLMM upper 16 bit mask is 0s
    gpio_config = GPIO_PIN_CFG(GPIO_INPUT,GPIO_PULL_UP,GPIO_STRENGTH_2MA,0);

    /* set gpio configuration for the gpio */
    if (GPIO_SUCCESS != gpio_set_config(fd, gpio_number_tlmm, 0, gpio_config)){
        mtouch_error(cypress->log_name,"gpio_set_config failed for gpio gpio_number %d \n", gpio_number);
        error_memory("Cypress_Touch: gpio_set_config failed for gpio gpio_number %d \n", gpio_number);
        goto exit_err;
    }

    /* set interrupt cfg for the gpio */
    if (GPIO_SUCCESS != gpio_set_interrupt_cfg(fd, gpio_number_tlmm, GPIO_INTERRUPT_TRIGGER_LOW, NULL)){
        mtouch_error(cypress->log_name,"gpio_set_config failed for gpio gpio_number %d \n", gpio_number);
        error_memory("Cypress_Touch: gpio_set_config failed for gpio gpio_number %d \n", gpio_number);
        goto exit_err;
    }

    /* get interrupt irq number from the configured interrupt gpio */
    if (GPIO_SUCCESS != gpio_get_interrupt_cfg(fd, gpio_number_tlmm, irq)){
        mtouch_error(cypress->log_name," failed to get irq for gpio gpio_number %d \n", gpio_number);
        error_memory ("Cypress_Touch: failed to get irq for gpio gpio_number %d \n", gpio_number);
        goto exit_err;
    }

    if (*irq == -1) {
        mtouch_error(cypress->log_name," irq corresponding to gpio %d is %d - exiting\n", gpio_number, *irq);
        error_memory("Cypress_Touch: irq corresponding to gpio %d is %d - exiting\n", gpio_number, *irq);
        goto exit_err;
    }
    mtouch_info(cypress->log_name,"irq for gpio_number %d is %d \n", gpio_number, *irq);
    return 0;

exit_err:
    return -1;
}

#if defined(__QNXNTO__) && defined(__USESRCVERSION)
#include <sys/srcversion.h>
__SRCVERSION("$URL: http://svn.ott.qnx.com/product/branches/7.0.0/trunk/hardware/mtouch/cypress/tp_intr.c $ $Rev: 886397 $")
#endif
